// Copyright  2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


using System;
using OneSpan.Image.Exceptions;
using OneSpan.Image.Generators;

namespace ImageGeneratorSDK_DotNet_Sample
{
    /// <summary>
    /// Sample class for the ImageGenerator
    /// </summary>
    static class ImageGeneratorSDKSample
    {
        // Number of pixels per square in Cronto image
        private const int CRONTO_IMAGE_SQUARE_SIZE = 6;

        // If the image is to be printed on paper
        private const bool CRONTO_IMAGE_ON_PAPER = true;

        // Length of the generated QRCode
        private const int QRCODE_IMAGE_SIZE = 210;

        // File name of the generated Cronto image
        private const string CRONTO_IMAGE_FILENAME = "crontoImage.png";

        // File name of the generated QRCode image
        private const string QRCODE_FILENAME = "qrCode.png";

        // Test string to generate the Cronto image
        private const string CRONTO_TEST_STRING = "5465737420537472696e6720666f7220496d61676547656e657261746f72";

        // Test string to generate the QRCode image
        private const string QRCODE_TEST_string = "Test string for ImageGenerator";

        private static ImageGeneratorSDK imageGeneratorSDK = new ImageGeneratorSDK();

        static void Main(string[] args)
        {
            Console.WriteLine("*************************************************************************\n");
            Console.WriteLine("*               Image Generator - Sample demonstration                  *\n");
            Console.WriteLine("*************************************************************************\n\n");

            // Generate Cronto image
            DemoGenerateDynamicCrontoImage();

            // Generate QR Code image
            DemoGenerateQRCode();
        }

        /// <summary>
        /// Example of how to generate a Cronto Image
        /// Generate a Cronto Image from the "Test String for ImageGenerator" String in its hexadecimal representation.
        ///
        /// The CRONTO images generated with the GenerateDynamicCrontoImage method cannot be scanned by VASCO hardware authenticators(e.g.DP760, DP780).
        /// 
        /// Use the GenerateCrontoSign method to generate CRONTO images compatible with VASCO hardware authenticators(e.g.DP760, DP780).
        /// </summary>
        private static void DemoGenerateDynamicCrontoImage()
        {
            try
            {
                Console.WriteLine("Generating Cronto image...");

                // Generate CrontoSign image
                using CrontoImageResponse crontoImg = imageGeneratorSDK.GenerateDynamicCrontoImage(CRONTO_IMAGE_SQUARE_SIZE, CRONTO_TEST_STRING, CRONTO_IMAGE_ON_PAPER);
                // Write output file
                crontoImg.SaveImage(CRONTO_IMAGE_FILENAME);

                Console.WriteLine("Cronto image generated : " + CRONTO_IMAGE_FILENAME);
            }
            catch (ImageGeneratorSDKException e)
            {
                Console.Error.WriteLine("Cronto image generation failed: " + e.Message);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine("Cronto image generation failed: " + e.Message);
            }
            Console.WriteLine("\n");
        }

        /// <summary>
        /// Example of how to generate a QR Code.
        /// Generate a QR Code from the "Test String for ImageGenerator" string.
        /// </summary>
        private static void DemoGenerateQRCode()
        {
            try
            {
                Console.WriteLine("Generating QR Code image...");

                // Generate QR Code
                using BWQRCodeResponse qrCodeResponse = imageGeneratorSDK.GenerateBWQRCode(QRCODE_IMAGE_SIZE,
                        QRCODE_TEST_string, ImageGeneratorSDKConstants.QRCODE_ERROR_CORRECTION_LEVEL_LOW);

                // Write output file
                qrCodeResponse.SaveImage(QRCODE_FILENAME);

                Console.WriteLine("QR Code image generated : " + QRCODE_FILENAME);
            }
            catch (ImageGeneratorSDKException e)
            {
                Console.Error.WriteLine("QR Code image generation failed: " + e.Message);
            }
            catch (Exception e)
            {
                Console.Error.WriteLine("QR Code image generation failed: " + e.Message);

            }
            Console.WriteLine("\n");
        }
    }
}
